package parser;

/**
 * Factory for T1a grammar non-terminal objects.
 */
public class T1aFact extends ATVFactory {
    /**
     * Factory to parse S grammar non-terminals.
     */
    private SFact _sFact;

    /**
     * Token visitor to parse S grammar non-terminals.
     */
    private ITokVisitor _parseS;


    /**
     * Initializer lambda for this factory.
     */
    private ILambda _initializer = new ILambda() {
        public Object apply(Object param) {
            // change state to no-op
            _initializer = NoOpLambda.Singleton;

            // initialize
            _parseS = _sFact.makeVisitor();
            return null;
        }
    };

    /**
     * Constructor for the T1a factory.
     *
     * @param tkz   tokenizer to use
     * @param sFact factory for S grammar non-terminals
     */
    public T1aFact(ITokenizer tkz, SFact sFact) {
        super(tkz);
        _sFact = sFact;
    }

    /**
     * Make the visitor.
     */
    private void initialize() {
        _initializer.apply(null);
    }

    /**
     * Constructor for the E1a factory, Does not take a factory for E grammar non-terminals. That factory has to be set
     * using setParseSFactory().
     *
     * @param tkz tokenizer to use
     */
    public T1aFact(ITokenizer tkz) {
        super(tkz);
    }

    /**
     * Set the factory for S grammar non-terminals.
     *
     * @param sFact factory for S grammar non-terminals
     */
    public void setParseSFactory(SFact sFact) {
        _sFact = sFact;
    }

    /**
     * Make a node.
     *
     * @param host mult token
     * @param inp  lambda to apply
     * @return produced node
     */
    private Object makeNode(MultToken host, Object inp) {
        ILambda l = (ILambda) inp;
        return l.apply(new T1a(host, (S) nextToken().execute(_parseS, inp)));
    }


    /**
     * Make a token visitor to parse an E1a non-terminal.
     *
     * @return token visitor
     */
    public ITokVisitor makeVisitor() {
        initialize();
        return new MultToken.IMultVisitor() {
            public Object multCase(MultToken host, Object inp) {
                return makeNode(host, inp);
            }

            public Object defaultCase(AToken host, Object param) {
                throw new IllegalArgumentException("Wrong token: '" + host + "'");
            }
        };
    }

    /**
     * Make a token visitor that delegates to the given visitor in a chain of responsibility.
     *
     * @param successor visitor to serve as successor in the chain
     */
    public ITokVisitor makeChainedVisitor(final ITokVisitor successor) {
        initialize();
        // We have a concrete token here, so we can determine a match immediately.
        // Therefore, we do not need to set up a lambda.
        return new MultToken.IMultVisitor() {
            public Object multCase(MultToken host, Object inp) {
                return makeNode(host, inp);
            }

            public Object defaultCase(AToken host, Object param) {
                return host.execute(successor, param);
            }
        };
    }

    /**
     * Make a token visitor that delegates to the given visitor in a chain of responsibility.
     *
     * @param successor       visitor to serve as successor in the chain
     * @param successorLambda lambda to pass to the successor in the default case
     */
    public ITokVisitor makeBacktrackChainedVisitor(final ITokVisitor successor, final ILambda successorLambda) {
        initialize();
        // We have a concrete token here, so we can determine a match immediately.
        // Therefore, we do not need to set up a lambda.
        return new MultToken.IMultVisitor() {
            public Object multCase(MultToken host, Object inp) {
                return makeNode(host, inp);
            }

            public Object defaultCase(AToken host, Object param) {
                return host.execute(successor, successorLambda);
            }
        };
    }
}

